
#ifndef SEQUENCENUMBER_H
#define SEQUENCENUMBER_H

#include <stdint.h>
#include <iostream>

// random
#include <boost/random/mersenne_twister.hpp>
#include <boost/random/uniform_int.hpp>

namespace ariba {
namespace overlay {

#define MIN_SEQ_NUM           100
#define SEQ_NUM_DISABLED        0

class SequenceNumber
{
public:
    static const SequenceNumber DISABLED;
    
    /// constructors
    SequenceNumber();
    SequenceNumber(uint32_t seqnum);
    SequenceNumber(uint64_t seqnum);
    
    /// factories
    static SequenceNumber createRandomSeqNum_Short();
    static SequenceNumber createRandomSeqNum_Long();


    /// get type information
    bool isShortSeqNum() const;
    bool isLongSeqNum() const;
    bool isDisabled() const;
    bool isValid() const;

    
    /// operators
    void increment();
    
    /// NOTE: this is also »true« if the long seq_num matches the short seq_num
    bool operator== (const SequenceNumber& rhs) const;
    
    /// NOTE: return value is only meaningful if »isValid() == true« for both values
    bool operator< (const SequenceNumber& rhs) const;

    bool isSuccessor(const SequenceNumber& rhs);
    bool isPredecessor(const SequenceNumber& rhs);
    
    
    /// getter
    uint32_t getShortSeqNum() const;
    uint64_t getLongSeqNum() const;

    
    /// to string
    friend std::ostream& operator<< (std::ostream& stream, const SequenceNumber& rhs);
    
private:
    uint32_t seqnum_32;
    uint64_t seqnum_64;

    /// static random_number_generator
    static boost::mt19937 rng_32bit;
    static boost::uniform_int<uint32_t> seqnum_distribution_32bit;
    
    // XXX NOTE: on later boost versions these are:
    //   boost::random::mt19937
    //   boost::random::uniform_int_distribution<>
        
    // NOTE: since the mt19937_64 is not available in this boost version, we 
    //   need an addition distribution
    static boost::uniform_int<uint32_t> distribution_full32bit;
};


}} // [namespace ariba::overlay]

#endif // SEQUENCENUMBER_H
