/*
 * The MIT License (MIT)
 * 
 * Copyright (c) 2013 Mario Hock mails2013@omnifile.org
 * 
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * 
 */

#include <iostream>
#include "ariba/utility/system/StartupWrapper.h"
#include <boost/property_tree/json_parser.hpp>
#include <stdexcept>

#include "Tunnel.h"
#include "Udp.h"

using ariba::utility::StartupWrapper;


/**
 * This program tunnels an (bidirectional) UDP connection over ARIBA:
 * 
 *  /------------/      UDP       /-------------/
 *  | UDP-Client |  <-----------> | AribaTunnel |
 *  /------------/               /--------------/
 *        .                              ^ 
 *        .                              | ARIBA
 *        .                              V
 *  /------------/      UDP       /-------------/
 *  | UDP-Server |  <-----------> | AribaTunnel |
 *  /------------/               /--------------/
 * 
 */
int main(int argc, char **argv)
{
    Tunnel tunnel;
    string config_file;
    
    // parameter: config file
    if ( argc == 2 )
    {
        config_file = argv[1];
        
        std::cout << "Using config: " << config_file << std::endl;
    }
    
    // read config
    ptree config;
    try
    {
        using boost::property_tree::json_parser::read_json;
        
        read_json(config_file, config);
    }
    catch ( std::exception& e )
    {
        logging_warn("ERROR: Failed to read config file »" << config_file << "«: ");
        logging_warn(e.what());
        logging_warn("---> Using fallback config.");
        
        config.put("ariba.spovnet_name", "tunnel");
    }

    
    // start System Queue
    StartupWrapper::startSystem();

    // ARIBA
    // TODO fix/update startup wrapper? -- but scheduleCall does basically the same
    SystemQueue::instance().scheduleCall(
        boost::bind(
                &Tunnel::init_ariba,
                &tunnel,
                config)
    );

    
    // ASIO (UDP)
    uint16_t listen_port = config.get<uint16_t>("tunnel.udp_listen_on_port", 0);
    uint16_t connect_port = config.get<uint16_t>("tunnel.udp_connect_to_port", 0);
    
    boost::asio::io_service io_service;
    udp_server server(io_service, tunnel);
    
    // client-side (use server-socket)
    if ( listen_port > 0 )
    {
        server.listen_on(listen_port);
    }
    // server side (use client_socket)
    else if ( connect_port > 0 )
    {
        server.connect_to(connect_port);
    }
    
    tunnel.register_udp_sender(&server);

    try
    {
        io_service.run();
    }
    catch (std::exception& e)
    {
        std::cerr << e.what() << std::endl;
    }
    
    // TODO stop tunnel gracefully.... fix/update startup wrapper?
    
    // stop System Queue
    StartupWrapper::stopSystem();
    
    return 0;
}
