/*
 * The MIT License (MIT)
 * 
 * Copyright (c) 2013 Mario Hock mails2013@omnifile.org
 * 
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * 
 */

#ifndef TUNNEL_H
#define TUNNEL_H

#include <ariba/ariba.h>
#include "MessageSenderInterface.h"

#include <boost/noncopyable.hpp>

using namespace ariba;
using boost::property_tree::ptree;

class Tunnel : 
    private boost::noncopyable,
    public ariba::NodeListener,
    public ariba::CommunicationListener,
    public MessageSenderInterface
{
public:
    Tunnel();
    virtual ~Tunnel();

    void init_ariba(ptree config);
    
    /*
     * Sends a message via ariba.
     *
     * This function is intended to be called from an EXTERNAL THREAD.
     * --> It synchronizes with the System Qeueue.
     */
    virtual void SendMessage(reboost::message_t msg);
    virtual void SendMessage(reboost::shared_buffer_t msg);
    
    void register_udp_sender(MessageSenderInterface* udp_sender);

protected:
    /* [ariba] communication listener interface */
    virtual void onMessage(reboost::shared_buffer_t message,
        const NodeID& remote,
        const LinkID& lnk,
        const SequenceNumber& seqnum,
        const ariba::overlay::OverlayMsg* overlay_msg);

    virtual void onLinkUp(const LinkID& lnk, const NodeID& remote);
    virtual void onLinkDown(const LinkID& lnk, const NodeID& remote);
//     virtual bool onLinkRequest(const NodeID& remote);
//     virtual void onLinkChanged(const LinkID& lnk, const NodeID& remote);
//     virtual void onLinkFail(const LinkID& lnk, const NodeID& remote);

    /* [ariba] node listener interface */
    virtual void onOverlayConnected( const SpoVNetID& vid );
    virtual void onOverlayDisconnected( const SpoVNetID& vid );

    
private:
    /* 
     * Sends a message via ariba.
     * 
     * This function must be called within the ARIBA THREAD!
     * It's the counterpart to Tunnel::SendMessage.
     */
    void send_message(reboost::message_t msg);
    
//     /// XXX this is only for debug output
//     void print_message(const string& prefix, reboost::message_t msg);
//     void print_message(const string& prefix, reboost::shared_buffer_t buf);

public:
    static const ariba::ServiceID TUNNEL_SERVICE_ID;
    
private:
    ariba::Node node;
    ariba::NodeID tunnel_endpoint;
    std::vector<LinkID> links;
    MessageSenderInterface* udp_sender;
};

#endif // TUNNEL_H
