#include "gtest/gtest.h"
#include "ariba/utility/system/SystemQueue.h"

#include <unistd.h>  // usleep

#include <ostream>

using namespace ::testing;
using namespace ariba::utility;
using namespace std;

/**
 *  Tests if the SystemQueue is initialized empty and not running.
 */
TEST(SystemQueue, Instantiation)
{
    SystemQueue& sysq = SystemQueue::instance();
    
//     cout << &sysq << endl;
    
    EXPECT_FALSE( sysq.isRunning() );
    EXPECT_TRUE( sysq.isEmpty() );
}


/**
 *  Tests whether calling the SystemQueue::instance() always returns the same object.
 * 
 *  NOTE: This is an easy case, since this is the same compile unit..
 *    But can't hurt to test it anyway.
 */
TEST(SystemQueue, Singleton)
{
    SystemQueue& sysq = SystemQueue::instance();
    SystemQueue& sysq2 = SystemQueue::instance();

//     cout << &sysq << endl;
    
    EXPECT_TRUE( &sysq == &sysq2 );
}


/**
 *  Start and stop the SystemQueue
 */
TEST(SystemQueue, StartStop)
{
    SystemQueue& sysq = SystemQueue::instance();

    EXPECT_FALSE( sysq.isRunning() );
    
    // start
    sysq.run();
    
    EXPECT_TRUE( sysq.isRunning() );
    
    // stop
    sysq.cancel();
    
    EXPECT_FALSE( sysq.isRunning() );
}


/**
 *  Test fixture
 * 
 *  class that can be called by the SystemQueue
 */
// To use a test fixture, derive a class from testing::Test.
class SystemQueueTest : 
        public testing::Test
{
public:

    SystemQueueTest() :
        checkmark(false)
    {
    }
        
    void Check()
    {
        checkmark = true;
//         cout << "Hallo Mario ^^" << endl;
    }

    bool checkmark;
};

TEST_F(SystemQueueTest, ScheduleCall)
{
    #define MAX_WAIT 100  // microseconds
    #define SLEEP_TIME 10  // microseconds

    SystemQueue& sysq = SystemQueue::instance();
    checkmark = false;  // just to be sure..
    
    // start
    sysq.run();
    
    // scheduleCall
    sysq.scheduleCall(
        boost::bind(&SystemQueueTest::Check, this)
    );
    
    // wait for the event to happen, (but not too long!)
    for ( int i = 0; i < MAX_WAIT / SLEEP_TIME; i++)
    {
        if ( checkmark )
            break;
        
        cout << "/// sleeping for " << SLEEP_TIME << " microseconds ..." << endl;
        usleep(SLEEP_TIME);
    }
    
    // stop
    sysq.cancel();
    
    EXPECT_TRUE( checkmark ) << "Function was called within " << MAX_WAIT << " microseconds.";    
}

