// [License]
// The Ariba-Underlay Copyright
//
// Copyright (c) 2008-2009, Institute of Telematics, Universität Karlsruhe (TH)
//
// Institute of Telematics
// Universität Karlsruhe (TH)
// Zirkel 2, 76128 Karlsruhe
// Germany
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE INSTITUTE OF TELEMATICS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE INSTITUTE OF TELEMATICS OR
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// The views and conclusions contained in the software and documentation
// are those of the authors and should not be interpreted as representing
// official policies, either expressed or implied, of the Institute of
// Telematics.
// [License]

#ifndef NAME_H_
#define NAME_H_

#include <iostream>
#include <memory.h>

// forward declaration
namespace ariba { class Name; }

std::ostream& operator<<( std::ostream&, const ariba::Name& );

#include "Identifiers.h"

namespace ariba {

/**
 * This class is a wrapper for canonical names.
 * Currently only human readable names are supported.
 *
 * @author Sebastian Mies <mies@tm.uka.de>
 */
class Name {
	friend std::ostream& operator<<( std::ostream&, const ::ariba::Name& );

private:
	bool _hreadable;
	bool _copy;
	int _length;
	uint8_t* _bytes;

	inline void init(const char* name, int len, bool copy, bool hreadable) {
		if (len == -1)
			len = strlen(name);

		if (copy) {
			_bytes = new uint8_t[len];
			memcpy(_bytes, name, len);
		} else {
			_bytes = (uint8_t*)name;
		}

		_copy = copy;
		_length = len;
		_hreadable = hreadable;
	}

public:
	static const Name UNSPECIFIED;

public:
	/**
	 * Constructs a new, yet unspecified name.
	 */
	inline Name() {
		_bytes = NULL;
		_length = 0;
		_copy = false;
		_hreadable = false;
	}

	/**
	 * Constructs a new name. If no length is specified, a human-readable
	 * name is assumed.
	 *
	 * @param name The name
	 * @param len The optional name length, if binary data is used as name
	 * @param copy A flag, whether the name's memory needs to be copied
	 */
	inline Name(const char* name, int len = -1, bool copy = false) {
		init(name, len, copy, len == -1);
	}

	/**
	 * Constructs a new name out of a human readable string.
	 *
	 * @param name A human readable name
	 */
	inline Name(std::string name) {
		init(name.c_str(), -1, true, true);
	}

	/**
	 * The copy constructor.
	 */
	inline Name(const Name& name) {
		init((const char*)name.bytes(), name.length(), true, name._hreadable);
	}

	/**
	 * Destroys the name and releases underlying memory if this name is a copy.
	 */
	inline ~Name() {
		if (_copy) delete _bytes;
	}

	/**
	 * Returns the binary bytes of the name
	 *
	 * @return The binary data
	 */
	inline const uint8_t* bytes() const {
		return _bytes;
	}

	/**
	 * Returns the length of the name in bytes.
	 *
	 * @return The length of the name
	 */
	inline const size_t length() const {
		return _length;
	}

	/**
	 * The common implementation of the "equal" operator.
	 */
	inline bool operator==(const Name& name) const {
		if (_bytes == NULL || name._bytes == NULL) return false;
		if (name.length() != length()) return false;
		return (memcmp(name.bytes(), bytes(), length()) == 0);
	}

	/**
	 * The common implementation of the "unequal" operator.
	 */
	inline bool operator!=(const Name& name) const {
		return !(*this == name);
	}

	/**
	 * Returns true, if the name is yet unspecified
	 */
	inline bool isUnspecified() {
		return *this == UNSPECIFIED;
	}

	/**
	 * Returns a random name.
	 */
	inline static Name random() {
		char name[17];
		for (int i=0;i<16; i++)
			name[i] = ("abcdefghijklmnopqrstuvwxyz")[((unsigned)rand())%26];
		name[16] = 0;
		return Name(name);
	}

	/**
	 * Returns a human-readable representation of this name
	 */
	inline std::string toString() const {
		if (_hreadable) {
			char str[256];
			for (int i=0; i<length(); i++) str[i] = bytes()[i];
			str[length()] = 0;
			return std::string(str);
		} else {
			return std::string("<not readable>");
		}
	}

	// hack: to be changed!
	inline NodeID toNodeId() const {
		return NodeID::sha1( bytes(), length() );
	}

	// hack: to be changed!
	inline SpoVNetID toSpoVNetId() const {
		return SpoVNetID::sha1(bytes(), length() );
	}
};

} // namespace ariba

#endif /* NAME_H_ */
