#include "PingPong.h"
#include "ariba/utility/configuration/Configuration.h"

using ariba::utility::Configuration;
using namespace ariba;

namespace ariba {
namespace application {
namespace pingpong {

// logging
use_logging_cpp( PingPong );

// the service id of the ping pong service
ServiceID PingPong::PINGPONG_ID = ServiceID( 111 );

// construction
PingPong::PingPong() : pingId( 0 ) {
}

// destruction
PingPong::~PingPong() {
}

// implementation of the startup interface
void PingPong::startup() {

	logging_info( "starting up PingPong service ... " );

	// create ariba module
	logging_debug( "creating ariba underlay module ... " );
	ariba = new AribaModule();

	// get the configuration object
	Configuration& config = Configuration::instance();

	// generate spovnet name
	Name spovnetName("pingpong");

	// get initiator flag
	this->isInitiator = Configuration::instance().read<bool> ("node.initiator");

	// get node name
	Name nodeName = Name::UNSPECIFIED;
	if (config.exists("node.name")) nodeName
			= config.read<string> ("node.name");

	// configure ariba module
	if (config.exists("ariba.ip.addr")) ariba->setProperty("ip.addr",
			config.read<string> ("ariba.ip.addr"));
	if (config.exists("ariba.tcp.port")) ariba->setProperty("tcp.port",
			config.read<string> ("ariba.tcp.port"));
	if (config.exists("ariba.udp.port")) ariba->setProperty("udp.port",
			config.read<string> ("ariba.udp.port"));
	if (config.exists("ariba.bootstrap.hints")) ariba->setProperty("bootstrap.hints",
			config.read<string> ("ariba.bootstrap.hints"));

	// start ariba module
	ariba->start();

	// create node and join
	node = new Node( *ariba, nodeName );

	// start node module
	node->start();

	// initiate or join the spovnet
	if (!isInitiator) node->join(spovnetName);
	else node->initiate(spovnetName);

	// bind communication and node listener
	node->bind(this);
	node->bind(this, PingPong::PINGPONG_ID);
	
	// ping pong started up...
	logging_info( "pingpong started up ");
}

// implementation of the startup interface
void PingPong::shutdown() {

	logging_info( "pingpong service starting shutdown sequence ..." );

	// stop timer
	Timer::stop();

	// unbind listeners
	node->unbind( this );
	node->unbind( this, PingPong::PINGPONG_ID );

	// leave spovnet
	node->leave();

	// stop the ariba module
	ariba->stop();

	// delete node and ariba module
	delete node;
	delete ariba;

	// now we are completely shut down
	logging_info( "pingpong service shut down" );
}

// node listener interface
void PingPong::onJoinCompleted( const SpoVNetID& vid ) {
	logging_info( "pingpong node join completed, spovnetid=" << vid.toString() );

	// start the timer to ping every second
	Timer::setInterval( 1000 );
	Timer::start();
}

void PingPong::onJoinFailed( const SpoVNetID& vid ) {
	logging_error(" pingpong node join failed, spovnetid=" << vid.toString() );
}

// communication listener
bool PingPong::onLinkRequest(const NodeID& remote, Message* msg) {
	return false;
}

void PingPong::onMessage(Message* msg, const NodeID& remote, 
		const LinkID& lnk = LinkID::UNSPECIFIED) {

	PingPongMessage* pingmsg = msg->decapsulate<PingPongMessage> ();

	logging_info( "received ping message on link " << lnk.toString()
			<< " from node " << remote.toString()
			<< ": " << pingmsg->toString() );
}

// timer event
void PingPong::eventFunction() {
	
	// we ping all nodes that are known in the overlay structure
	// this can be all nodes (OneHop) overlay or just some neighbors
	// in case of a Chord or Kademlia structure
	
	logging_info( "pinging overlay neighbors with ping id " << ++pingId );

	PingPongMessage pingmsg( pingId );
	node->sendBroadcastMessage( pingmsg, PingPong::PINGPONG_ID );

}

}}} // namespace ariba, application, pingpong
