// [License]
// The Ariba-Underlay Copyright
//
// Copyright (c) 2008-2009, Institute of Telematics, Universität Karlsruhe (TH)
//
// Institute of Telematics
// Universität Karlsruhe (TH)
// Zirkel 2, 76128 Karlsruhe
// Germany
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE INSTITUTE OF TELEMATICS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE ARIBA PROJECT OR
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// The views and conclusions contained in the software and documentation
// are those of the authors and should not be interpreted as representing
// official policies, either expressed or implied, of the Institute of
// Telematics.
// [License]

#include "EndpointDescriptor.h"
#include "ariba/utility/misc/StringFormat.h"
#include "ariba/utility/misc/Helper.h"

namespace ariba {
namespace communication {

vsznDefault(EndpointDescriptor);

const EndpointDescriptor EndpointDescriptor::UNSPECIFIED;

EndpointDescriptor::EndpointDescriptor() : locator( NULL ), isUnspec( true ){
}

EndpointDescriptor::EndpointDescriptor(const EndpointDescriptor& rh){
	locator = (rh.locator != NULL) ? new IPv4Locator(*rh.locator) : NULL;
	isUnspec = rh.isUnspec;
}

EndpointDescriptor::EndpointDescriptor(const Locator* _locator){
	if( _locator == NULL ) return;

	locator = new IPv4Locator(*dynamic_cast<IPv4Locator*>((Locator*)_locator));
	isUnspec = false;
}

EndpointDescriptor::EndpointDescriptor(const string str) {
	using namespace boost::xpressive;
	using namespace ariba::utility::string_format;
	using namespace ariba::utility::Helper;
	using namespace std;

	locator = NULL;
	isUnspec = true;

	smatch match;
	if (regex_search(str, match, robjects)) {
		regex_nav nav = match;
		for (int i=0; i<nav.size(); i++) {
			string type = nav[i][robject_id].str();
			if (type=="ip") {
				string ip = nav[i][robject_data].str();
				ip = ip.substr(1,ip.size()-2);
				this->locator = new IPv4Locator();
				this->locator->setIP(ip);
				this->isUnspec = false;
			} else
			if (type=="tcp") {
				string port = nav[i][robject_data][rfields][1].str();
				port = port.substr(1,port.size()-2);
				this->locator->setPort(stoi(port));
			}
		}
	}
}

EndpointDescriptor::~EndpointDescriptor() {
}

bool EndpointDescriptor::isUnspecified() const {
	return isUnspec;
}

string EndpointDescriptor::toString() const {
	if( locator == NULL ) return "<undefined locator>";
	std::ostringstream o;
	o << "ip{" << locator->getIP() << "}";
	o << ",";
	o << "tcp(ip,{" << locator->getPort() << "})";
	return o.str();
}

EndpointDescriptor* EndpointDescriptor::fromString( string str ) {
	return new EndpointDescriptor( str );
}

bool EndpointDescriptor::operator!=( const EndpointDescriptor& rh ) const {
	return !operator==(rh);
}

bool EndpointDescriptor::operator==(const EndpointDescriptor& rh) const {

	if( isUnspecified() && rh.isUnspecified() ) {

		// both unspec bit set
		return true;

	} else if( (!isUnspecified()) && (!rh.isUnspecified()) ) {

		//
		// both are valid, check locators
		//

		if( locator == NULL && rh.locator == NULL ){

			// both locators are invalid, ok true
			return true;

		} else if( locator == NULL ^ rh.locator == NULL ) {

			// one locator is invalid, the other not, false
			return false;

		} else {

			// both locators are valid, compare
			assert( locator != NULL && rh.locator != NULL );
			return ( locator->operator==(*rh.locator) );

		}

	} else {

		// one is unspec, the other not
		assert( isUnspecified() ^ rh.isUnspecified() );
		return false;

	}

}

}} // namespace ariba, communication
