// [License]
// The Ariba-Underlay Copyright
//
// Copyright (c) 2008-2009, Institute of Telematics, Universität Karlsruhe (TH)
//
// Institute of Telematics
// Universität Karlsruhe (TH)
// Zirkel 2, 76128 Karlsruhe
// Germany
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE INSTITUTE OF TELEMATICS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE ARIBA PROJECT OR
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// The views and conclusions contained in the software and documentation
// are those of the authors and should not be interpreted as representing
// official policies, either expressed or implied, of the Institute of
// Telematics.
// [License]

#ifndef OVERLAY_MSG_H__
#define OVERLAY_MSG_H__

#include <boost/cstdint.hpp>
#include "ariba/utility/messages.h"
#include "ariba/utility/serialization.h"
#include "ariba/utility/types/ServiceID.h"
#include "ariba/utility/types/NodeID.h"
#include "ariba/utility/types/LinkID.h"

using ariba::utility::LinkID;
using ariba::utility::NodeID;
using ariba::utility::ServiceID;
using ariba::utility::Message;

namespace ariba {
namespace overlay {

using_serialization
;

class OverlayMsg: public Message {
VSERIALIZEABLE
	;
public:

	/// (payload-) message types
	enum type_ {
		typeInvalid = 0, ///< invalid type (no encapsulated messages)
		typeData = 1, ///< message contains data for higher layers
		typeJoinRequest = 2, ///< join request
		typeJoinReply = 3, ///< join reply
		typeUpdate = 4, ///< update message for link association
		typeLinkRequest = 5, ///< link request (sent over the overlay)
		typeRelay = 6, ///< relay message
		typeKeepAlive = 7, ///< a keep-alive message
		typeDirectLink = 8,
	///< a direct connection has been established
	};

	/// default constructor
	OverlayMsg(type_ type = typeInvalid, const ServiceID _service =
			ServiceID::UNSPECIFIED, const NodeID _sourceNode =
			NodeID::UNSPECIFIED) :
		type((uint8_t) type), service(_service), sourceNode(_sourceNode),
				relayLink(LinkID::UNSPECIFIED), autoLink(false) {
	}

	OverlayMsg(const OverlayMsg& rhs) :
		type(rhs.type), service(rhs.service), sourceNode(rhs.sourceNode),
				relayLink(rhs.relayLink), autoLink(rhs.autoLink) {
	}

	/// type and source node constructor
	OverlayMsg(type_ type, const NodeID _sourceNode) :
		type((uint8_t) type), service(ServiceID::UNSPECIFIED), sourceNode(
				_sourceNode), relayLink(LinkID::UNSPECIFIED), autoLink(false) {
	}

	/// destructor
	~OverlayMsg();

	type_ getType() const {
		return (type_) type;
	}

	const ServiceID& getService() const {
		return service;
	}

	const NodeID& getSourceNode() const {
		return sourceNode;
	}

	const LinkID& getRelayLink() const {
		return relayLink;
	}

	void setRelayLink(const LinkID& relayLink) {
		this->relayLink = relayLink;
	}

	const bool isAutoLink() const {
		return autoLink;
	}

	void setAutoLink(bool autoLink) {
		this->autoLink = autoLink;
	}
private:
	uint8_t type;
	ServiceID service;
	NodeID sourceNode;
	LinkID relayLink;
	uint8_t autoLink;
};

}
} // ariba::overlay

sznBeginDefault( ariba::overlay::OverlayMsg, X ){
X && type && &service && &sourceNode;
if (type == typeDirectLink) X && &relayLink;
if (type == typeUpdate) X && autoLink;
X && Payload();
}sznEnd();

#endif // OVERLAY_MSG_H__
