// [License]
// The Ariba-Underlay Copyright
//
// Copyright (c) 2008-2009, Institute of Telematics, Universität Karlsruhe (TH)
//
// Institute of Telematics
// Universität Karlsruhe (TH)
// Zirkel 2, 76128 Karlsruhe
// Germany
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE INSTITUTE OF TELEMATICS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE ARIBA PROJECT OR
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// The views and conclusions contained in the software and documentation
// are those of the authors and should not be interpreted as representing
// official policies, either expressed or implied, of the Institute of
// Telematics.
// [License]

#include "LinkID.h"

#include <boost/unordered_map.hpp>

namespace ariba {
namespace utility {

/// the unspecified link id
const LinkID LinkID::UNSPECIFIED;

const char* UNSPECIFIED_LINK 	= "<LINKID-UNSPECIFIED>";
const char* UNKNOWN_LINK 		= "<LINKID-UNKNOWN>";
const char* NULL_INFO 			= "<NO-INFO-AVAILABLE>";

boost::unordered_map<uint16_t, const char*> link_ids;

bool LinkID::isValid( const LinkID& id ) {
	return link_ids.count(id.local_id)!=0;
}

const char* LinkID::getInfo( const LinkID& id ) {
	if (!id.valid())
		return UNSPECIFIED_LINK;
	if ( link_ids.count(id.local_id) == 0 )
		return UNKNOWN_LINK;
	const char* info = link_ids.find( id.local_id )->second;
	if (info == NULL)
		return NULL_INFO;
	return info;
}

/// create a new locally unique link id
LinkID LinkID::create( const char* info ) {
	assert( link_ids.size() != 0xFFFE );
	uint16_t id;
	do {
		id = rand() & 0xFFFF;
	} while (id == 0 || link_ids.count(id) != 0);
	link_ids.insert( std::make_pair(id, info) );
	return LinkID(id);
}

/// free a locally unique link id
void LinkID::destroy( const LinkID& id ) {
	link_ids.erase(id.local_id);
}

std::ostream& operator<<(std::ostream& s, const LinkID& id ) {
	return s << id.toString();
}

}} // spovnet, common
